<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Teams extends Home_Controller {
    public function __construct() {
        parent::__construct();
        $this->load->model('Teams_model');
        $this->load->library(['session', 'pagination']);
        if (!$this->session->userdata('logged_in')) {
            redirect('admin/login');
        }
    }

    public function save_teams_settings() {
        $data = [
            'client_id' => $this->input->post('client_id'),
            'client_secret' => $this->input->post('client_secret'),
            'tenant_id' => $this->input->post('tenant_id')
        ];
        $this->Teams_model->save_settings($data);
        
        if ($this->input->is_ajax_request()) {
            $response = [
                'status' => 'success',
                'message' => 'Microsoft Teams settings saved successfully!'
            ];
            echo json_encode($response);
            exit;
        }
        
        redirect('admin/settings');
    }

    public function teams_auth() {
        $client_id = $this->Teams_model->get_settings()->client_id;
        $redirect_uri = base_url('admin/teams/teams_callback');
        
        $scopes = [
            'offline_access',
            'OnlineMeetings.ReadWrite',
            'Calendars.ReadWrite',
            'User.Read'
        ];
        
        $encoded_scopes = urlencode(implode(' ', $scopes));
        $auth_url = "https://login.microsoftonline.com/common/oauth2/v2.0/authorize?client_id=$client_id&response_type=code&redirect_uri=$redirect_uri&scope=$encoded_scopes&prompt=consent";
        
        error_log("Redirecting to Microsoft auth: " . $auth_url);
        redirect($auth_url);
    }

    public function teams_callback() {
       try {
            $code = $this->input->get('code');
            if (empty($code)) {
                throw new Exception('No authorization code received from Microsoft.');
            }
            
            error_log("Teams callback received code: " . substr($code, 0, 10) . "...");
            
            $settings = $this->Teams_model->get_settings();
            
            $client = new GuzzleHttp\Client([
                'timeout' => 30,
                'connect_timeout' => 30
            ]);
            
            $redirect_uri = base_url('admin/teams/teams_callback');
            error_log("Using redirect URI: " . $redirect_uri);
            
            $response = $client->post('https://login.microsoftonline.com/common/oauth2/v2.0/token', [
                'form_params' => [
                    'client_id' => $settings->client_id,
                    'client_secret' => $settings->client_secret,
                    'grant_type' => 'authorization_code',
                    'code' => $code,
                    'redirect_uri' => $redirect_uri,
                    'scope' => 'offline_access OnlineMeetings.ReadWrite Calendars.ReadWrite User.Read'
                ]
            ]);
            
            $token_data = json_decode($response->getBody(), true);
            
            if (!isset($token_data['access_token'])) {
                throw new Exception('Invalid response from Microsoft. No access token received.');
            }
            
            error_log("Teams auth successful - token received with expiry: " . $token_data['expires_in'] . " seconds");
            $this->Teams_model->save_tokens($token_data);
            
            $verify_client = new GuzzleHttp\Client(['timeout' => 10]);
            $verify_response = $verify_client->get('https://graph.microsoft.com/v1.0/me', [
                'headers' => ['Authorization' => 'Bearer ' . $token_data['access_token']]
            ]);
            
            $user_data = json_decode($verify_response->getBody(), true);
            error_log("Token verified successfully. Connected as: " . $user_data['displayName']);
            
            $this->session->set_flashdata('success', 'Successfully connected to Microsoft Teams as ' . $user_data['displayName'] . '!');
            redirect('admin/teams/connection_status');
        } catch (Exception $e) {
            echo '<pre>';
            print_r($e);
            echo '</pre>';
            die();
            error_log("Teams auth error: " . $e->getMessage());
            
            $error_message = $e->getMessage();
            if (method_exists($e, 'getResponse') && $e->getResponse()) {
                $response_body = $e->getResponse()->getBody()->getContents();
                error_log("Teams auth error response: " . $response_body);
                
                $error_data = json_decode($response_body, true);
                if (isset($error_data['error_description'])) {
                    $error_message .= ': ' . $error_data['error_description'];
                }
            }
            
            $this->session->set_flashdata('error', 'Error authenticating with Microsoft Teams: ' . $error_message);
            redirect('admin/teams/connection_status');
        }
    }

    public function connection_status() {
        $data = array();
        $data['page_title'] = 'Microsoft Teams Connection Status';
        $data['page'] = 'Microsoft Teams';
        
        try {
            $settings = $this->Teams_model->get_settings();
            
            if (!$settings) {
                $data['status'] = 'not_configured';
                $data['message'] = 'Microsoft Teams is not configured. Please add your API credentials in settings.';
                $data['details'] = [];
                $this->session->set_flashdata('error', 'Microsoft Teams is not configured.');
            } elseif (empty($settings->access_token) || empty($settings->refresh_token)) {
                $data['status'] = 'not_authenticated';
                $data['message'] = 'API credentials are set, but you need to authenticate with Microsoft.';
                $data['details'] = [
                    'client_id' => $settings->client_id ? ('Configured: ' . substr($settings->client_id, 0, 8) . '...') : 'Missing',
                    'client_secret' => $settings->client_secret ? 'Configured' : 'Missing',
                    'tenant_id' => isset($settings->tenant_id) ? $settings->tenant_id : 'Using common endpoint'
                ];
                $this->session->set_flashdata('error', 'Please authenticate with Microsoft Teams first.');
            } else {
                $token = $this->Teams_model->get_access_token();
                
                if (!$token) {
                    $data['status'] = 'token_error';
                    $data['message'] = 'Unable to get a valid access token. Please re-authenticate.';
                    $data['details'] = [
                        'token_expiry' => isset($settings->token_expiry) ? $settings->token_expiry : 'Unknown',
                        'access_token_length' => strlen($settings->access_token),
                        'refresh_token_length' => strlen($settings->refresh_token)
                    ];
                    $this->session->set_flashdata('error', 'Authentication token error. Please reconnect.');
                } else {
                    $client = new GuzzleHttp\Client(['timeout' => 10]);
                    $response = $client->get('https://graph.microsoft.com/v1.0/me', [
                        'headers' => ['Authorization' => 'Bearer ' . $token]
                    ]);
                    
                    $user_data = json_decode($response->getBody(), true);
                    
                    $data['status'] = 'connected';
                    $data['message'] = 'Successfully connected to Microsoft Teams API!';
                    $data['details'] = [
                        'user' => isset($user_data['displayName']) ? $user_data['displayName'] : 'Unknown',
                        'email' => isset($user_data['userPrincipalName']) ? $user_data['userPrincipalName'] : 'Unknown',
                        'token_expiry' => date('Y-m-d H:i:s', strtotime($settings->token_expiry)),
                        'time_remaining' => floor((strtotime($settings->token_expiry) - time()) / 60) . ' minutes',
                        'scopes' => implode(', ', [
                            'offline_access',
                            'OnlineMeetings.ReadWrite',
                            'Calendars.ReadWrite',
                            'User.Read'
                        ])
                    ];
                    
                    try {
                        $presence_response = $client->get('https://graph.microsoft.com/v1.0/me/presence', [
                            'headers' => ['Authorization' => 'Bearer ' . $token]
                        ]);
                        $presence_data = json_decode($presence_response->getBody(), true);
                        if (isset($presence_data['availability'])) {
                            $data['details']['presence'] = $presence_data['availability'];
                        }
                    } catch (Exception $presence_ex) {
                        $data['details']['presence_error'] = 'Unable to get presence: ' . $presence_ex->getMessage();
                    }
                    
                    $this->session->set_flashdata('success', 'Microsoft Teams connection is working properly.');
                }
            }
        } catch (Exception $e) {
            $data['status'] = 'error';
            $data['message'] = 'Error testing connection: ' . $e->getMessage();
            $data['details'] = [];
            
            if (method_exists($e, 'getResponse') && $e->getResponse()) {
                $response_body = $e->getResponse()->getBody()->getContents();
                $error_data = json_decode($response_body, true);
                if (isset($error_data['error']['message'])) {
                    $data['details']['api_error'] = $error_data['error']['message'];
                } else {
                    $data['details']['response'] = $response_body;
                }
            }
            
            $this->session->set_flashdata('error', 'Connection error: ' . $e->getMessage());
        }
        
        $data['main_content'] = $this->load->view('admin/teams_status', $data, TRUE);
        $this->load->view('admin/index', $data);
    }
    
    public function reset_connection() {
        $this->db->update('teams_settings', ['access_token' => '', 'refresh_token' => '', 'token_expiry' => NULL]);
        $this->session->set_flashdata('success', 'Microsoft Teams connection has been reset. Please reconnect your account.');
        redirect('admin/teams/connection_status');
    }

    public function create_meeting($booking_id)
    {
        $this->db->where('id', $booking_id);
        $this->db->update('session_booking', ['meeting_type' => 'teams']);

        try {
            $booking = $this->admin_model->get_by_id($booking_id, 'session_booking');
            if (!$booking) throw new Exception('Booking not found');

            $session = $this->admin_model->get_by_id($booking->session_id, 'sessions');
            $mentor = $this->admin_model->get_by_id($booking->user_id, 'users');
            $mentee = $this->admin_model->get_by_id($booking->mentee_id, 'users');

            $settings = $this->Teams_model->get_settings();
            if (!$settings || empty($settings->client_id) || empty($settings->client_secret)) {
                throw new Exception('Microsoft Teams not configured. Please add your API credentials in settings.');
            }

            try {
                $access_token = $this->Teams_model->get_access_token();
            } catch (Exception $e) {
                $this->session->set_flashdata('info', 'Unable to create meeting. Please try again after some time.');
                redirect('admin/sessions/booking');
            }

            $time = explode('-', $booking->time);
            if (count($time) && $booking->date != null) {
                $time1 = trim($time[0]);
                $time2 = trim($time[1]);
                $date = $booking->date;
            } else {
                throw new Exception('Invalid booking time format');
            }

            $start_datetime = new DateTime($date . ' ' . $time1, new DateTimeZone(date_default_timezone_get()));
            $end_datetime = new DateTime($date . ' ' . $time2, new DateTimeZone(date_default_timezone_get()));
            $start_datetime->setTimezone(new DateTimeZone('UTC'));
            $end_datetime->setTimezone(new DateTimeZone('UTC'));
            $formatted_start = $start_datetime->format('Y-m-d\TH:i:s');
            $formatted_end = $end_datetime->format('Y-m-d\TH:i:s');

            $meeting_subject = "{$session->name} - {$booking->booking_number}";

            $request_body = [
                'subject' => $meeting_subject,
                'start' => [
                    'dateTime' => $formatted_start,
                    'timeZone' => 'UTC'
                ],
                'end' => [
                    'dateTime' => $formatted_end,
                    'timeZone' => 'UTC'
                ],
                'location' => [
                    'displayName' => 'Microsoft Teams Meeting'
                ],
                'attendees' => [
                    [
                        'emailAddress' => [
                            'address' => $mentor->email,
                            'name' => $mentor->name
                        ],
                        'type' => 'required'
                    ],
                    [
                        'emailAddress' => [
                            'address' => $mentee->email,
                            'name' => $mentee->name
                        ],
                        'type' => 'required'
                    ]
                ],
                'isOnlineMeeting' => true,
                'onlineMeetingProvider' => 'teamsForBusiness'
            ];

            $client = new GuzzleHttp\Client(['timeout' => 30]);

            try {
                $response = $client->post("https://graph.microsoft.com/v1.0/me/events", [
                    'headers' => [
                        'Authorization' => 'Bearer ' . $access_token,
                        'Content-Type' => 'application/json'
                    ],
                    'json' => $request_body
                ]);

                $meeting_data = json_decode($response->getBody(), true);

                $join_url = $meeting_data['onlineMeeting']['joinUrl'] ?? '';
                $web_url = $meeting_data['webLink'] ?? '';

                $teams_meeting = [
                    'booking_id' => $booking_id,
                    'teams_meeting_id' => $meeting_data['id'],
                    'join_url' => $join_url,
                    'join_web_url' => $web_url,
                    'subject' => $meeting_data['subject'],
                    'creation_datetime' => date('Y-m-d H:i:s'),
                    'meeting_start' => $formatted_start,
                    'meeting_end' => $formatted_end
                ];

                $this->Teams_model->save_meeting($teams_meeting);

                $this->db->where('id', $booking_id);
                $this->db->update('session_booking', [
                    'join_url' => $join_url,
                    'host_url' => $web_url ?: $join_url,
                    'is_start' => 0
                ]);

              $this->session->set_flashdata('msg', 'Microsoft Teams meeting created successfully and invitations have been emailed to all participants.');
                redirect('admin/sessions/booking');

            } catch (GuzzleHttp\Exception\ClientException $e) {
                $response_body = $e->getResponse() ? $e->getResponse()->getBody()->getContents() : 'No response body';
                $error_data = json_decode($response_body, true);
                $error_message = $error_data['error']['message'] ?? 'API Error: ' . $e->getMessage();

                $this->session->set_flashdata('error', 'Oops! Error while creating Teams meeting: ' . $error_message . ' Please try again after some time.');

                redirect('admin/sessions/booking');

            } catch (Exception $e) {
                $this->session->set_flashdata('error', 'Oops! Error while creating Teams meeting: ' . $e->getMessage() . ' Please try again after some time.');
                redirect('admin/sessions/booking');
            }

        } catch (Exception $e) {
            $this->session->set_flashdata('error', 'Oops! Error while creating Teams meeting: ' . $e->getMessage() . ' Please try again after some time.');
            redirect('admin/sessions/booking');
        }
    }

    
    public function start_meeting($booking_id) {
        if (!$this->session->userdata('logged_in')) {
            redirect('admin/login');
        }
        
        $edit_data = [
            'is_start' => 1
        ];

        if ($booking_id != 0) {
            $this->admin_model->edit_option($edit_data, $booking_id, 'session_booking');
        }
        redirect($_SERVER['HTTP_REFERER']);
    }
    
    public function cancel_meeting($booking_id) 
    {
        if(!$this->session->userdata('logged_in')){
            redirect(base_url('auth'));
        }
        
        $booking = $this->admin_model->get_by_id($booking_id, 'session_booking');
        
        if (!empty($booking)) {
            $teams_meeting = $this->db->get_where('teams_meetings', ['booking_id' => $booking_id])->row();
            
            if ($teams_meeting) {
                try {
                    $access_token = $this->Teams_model->get_access_token();
                    $client = new GuzzleHttp\Client(['timeout' => 30]);
                    
                    $client->delete("https://graph.microsoft.com/v1.0/me/events/{$teams_meeting->meeting_id}", [
                        'headers' => [
                            'Authorization' => "Bearer $access_token"
                        ]
                    ]);
                    
                    $this->db->where('id', $booking_id);
                    $this->db->update('session_booking', [
                        'is_start' => 0,
                        'host_url' => '',
                        'join_url' => '',
                        'meeting_type' => null
                    ]);

                    $this->db->where('id', $teams_meeting->id);
                    $this->db->delete('teams_meetings');

                    $this->session->set_flashdata('msg', 'Microsoft Teams meeting cancelled successfully.');
                    
                } catch (Exception $e) {
                    $this->session->set_flashdata('error', 'Oops! Failed to cancel meeting on Microsoft Teams: ' . $e->getMessage());
                }
            }
        }
        
        redirect($_SERVER['HTTP_REFERER']);
    }
}